'use client';

import { Event } from '@/types/event';
import { motion } from 'framer-motion';
import { format, parseISO } from 'date-fns';
import { Calendar, MapPin, Users, ArrowRight, Sparkles, Ticket } from 'lucide-react';
import Link from 'next/link';

interface FeaturedEventsProps {
  events: Event[];
}

/**
 * FeaturedEvents component - White cards with red border accent
 */
export default function FeaturedEvents({ events }: FeaturedEventsProps) {
  if (events.length === 0) return null;

  return (
    <section id="featured" className="py-12 bg-white">
      <div className="flex items-center justify-between mb-8">
        <div>
          <div className="flex items-center gap-2 mb-2">
            <Sparkles className="w-5 h-5 text-[#E11D2E]" />
            <span className="text-sm font-semibold text-[#E11D2E] uppercase tracking-wider">
              Featured
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl font-heading font-semibold text-gray-900">
            Trending Events
          </h2>
        </div>
        <Link 
          href="/" 
          className="hidden sm:flex items-center gap-2 text-sm text-[#E11D2E] hover:text-[#BE123C] font-medium transition-colors"
        >
          View All <ArrowRight className="w-4 h-4" />
        </Link>
      </div>

      {/* Top row - 3 compact cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
        {events.slice(0, 3).map((event, index) => (
          <motion.div
            key={event.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5, delay: index * 0.1 }}
          >
            <Link href={`/event/${event.id}`}>
              <div className="bg-white rounded-2xl overflow-hidden border-2 border-gray-200 hover:border-[#E11D2E] shadow-[0_4px_20px_rgba(0,0,0,0.04)] hover:shadow-[0_12px_40px_rgba(225,29,46,0.12)] transition-all duration-300 group h-full">
                {/* Content section */}
                <div className="p-5 relative">
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex items-center gap-2">
                      <span className="inline-flex items-center px-2.5 py-1 rounded-full bg-[#E11D2E] text-white text-[10px] font-bold uppercase tracking-wide">
                        {event.eventType.replace('-', ' ')}
                      </span>
                      <span className="text-xs text-gray-500 font-medium">
                        {event.country}
                      </span>
                    </div>
                    <div className="flex flex-col items-center px-3 py-2 rounded-lg bg-[#FFF5F5] border border-red-100">
                      <span className="text-[10px] text-[#E11D2E] uppercase font-semibold">
                        {format(parseISO(event.date), 'MMM')}
                      </span>
                      <span className="text-xl font-bold text-[#E11D2E]">
                        {format(parseISO(event.date), 'd')}
                      </span>
                    </div>
                  </div>
                  <h3 className="font-heading text-xl font-semibold text-gray-900 mb-2 line-clamp-1 group-hover:text-[#E11D2E] transition-colors">
                    {event.name}
                  </h3>
                  <div className="flex items-center gap-2 text-gray-500 text-sm">
                    <Users className="w-4 h-4 text-[#E11D2E]" />
                    <span>{event.attendeesRange} attendees</span>
                  </div>
                </div>
                
                {/* Footer */}
                <div className="px-5 py-4 flex items-center justify-between bg-gray-50 border-t border-gray-100">
                  {event.ticketAvailable ? (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-[#FEE2E2] text-[#E11D2E] text-xs font-semibold uppercase border border-[#FECACA]">
                      <Ticket className="w-3 h-3" />
                      Tickets Available
                    </span>
                  ) : (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-gray-100 text-gray-600 text-xs font-semibold uppercase border border-gray-200">
                      Invite Only
                    </span>
                  )}
                  <span className="text-sm font-semibold text-gray-900">
                    {event.pricing || 'Free Entry'}
                  </span>
                </div>
              </div>
            </Link>
          </motion.div>
        ))}
      </div>

      {/* Bottom row - 2 larger cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {events.slice(3, 5).map((event, index) => (
          <motion.div
            key={event.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5, delay: 0.3 + index * 0.1 }}
          >
            <Link href={`/event/${event.id}`}>
              <div className="bg-white rounded-2xl overflow-hidden border-2 border-gray-200 hover:border-[#E11D2E] shadow-[0_4px_20px_rgba(0,0,0,0.04)] hover:shadow-[0_12px_40px_rgba(225,29,46,0.12)] transition-all duration-300 group">
                {/* Content section */}
                <div className="p-6 relative">
                  <div className="flex gap-4">
                    {/* Date badge */}
                    <div className="flex-shrink-0 w-16 h-16 rounded-xl bg-[#FFF5F5] border border-red-100 flex flex-col items-center justify-center">
                      <span className="text-xs text-[#E11D2E] uppercase font-semibold">
                        {format(parseISO(event.date), 'MMM')}
                      </span>
                      <span className="text-2xl font-bold text-[#E11D2E]">
                        {format(parseISO(event.date), 'd')}
                      </span>
                    </div>
                    
                    {/* Content */}
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="inline-flex items-center px-2.5 py-1 rounded-full bg-[#E11D2E] text-white text-[10px] font-bold uppercase tracking-wide">
                          {event.eventType.replace('-', ' ')}
                        </span>
                        <span className="text-xs text-gray-500 font-medium">
                          {event.country}
                        </span>
                      </div>
                      <h4 className="font-heading text-xl font-semibold text-gray-900 mb-2 group-hover:text-[#E11D2E] transition-colors">
                        {event.name}
                      </h4>
                      <p className="text-gray-600 text-sm line-clamp-2 mb-3">
                        {event.description}
                      </p>
                      <div className="flex flex-wrap items-center gap-4 text-sm text-gray-500">
                        <span className="flex items-center gap-1">
                          <Calendar className="w-4 h-4 text-[#E11D2E]" />
                          {format(parseISO(event.date), 'MMM d, yyyy')}
                          {event.endDate && ` - ${format(parseISO(event.endDate), 'MMM d, yyyy')}`}
                        </span>
                        <span className="flex items-center gap-1">
                          <MapPin className="w-4 h-4 text-[#E11D2E]" />
                          {event.venue}, {event.city}
                        </span>
                        <span className="flex items-center gap-1">
                          <Users className="w-4 h-4 text-[#E11D2E]" />
                          {event.attendeesRange} attendees
                        </span>
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* Footer */}
                <div className="px-6 py-4 flex items-center justify-between bg-gray-50 border-t border-gray-100">
                  {event.ticketAvailable ? (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-[#FEE2E2] text-[#E11D2E] text-xs font-semibold uppercase border border-[#FECACA]">
                      <Ticket className="w-3 h-3" />
                      Tickets Available
                    </span>
                  ) : (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-gray-100 text-gray-600 text-xs font-semibold uppercase border border-gray-200">
                      Invite Only
                    </span>
                  )}
                  <span className="text-sm font-semibold text-gray-900">
                    {event.pricing || 'Invitation only'}
                  </span>
                </div>
              </div>
            </Link>
          </motion.div>
        ))}
      </div>
    </section>
  );
}
